/*
 * Kuklomenos
 * Copyright (C) 2008-2009 Martin Bays <mbays@sdf.lonestar.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/.
 */

#include <config.h>

#ifdef HAVE_LIBCURL
#define HIGH_SCORE_REPORTING 1
#endif

#include <cstdlib>
#include <cstdio>
#include <ctime>
#include <stack>
#include <SDL/SDL.h>
#include <SDL_gfxPrimitivesDirty.h>
#include <string>
#include <sstream>

#include "settings.h"
#include "geom.h"
#include "clock.h"
#include "state.h"
#include "data.h"
#include "background.h"

SDL_Surface* screen = NULL;

enum EventsReturn
{
    ER_NONE,
    ER_MISC,
    ER_QUIT,
    ER_SURRENDER,
    ER_RESTART,
    ER_SCREENSHOT,
    ER_NEWBACKGROUND,
    ER_NOTIMETAKEN,
    ER_MENU
};
bool setVideoMode()
{
    SDL_Surface* ret = SDL_SetVideoMode(settings.width, settings.height,
	    settings.bpp, settings.videoFlags);

    if (!ret)
    {
	settings.width = screen->w;
	settings.height = screen->h;
	settings.bpp = screen->format->BitsPerPixel;
	return false;
    }

    screen = ret;
    screenGeom = ScreenGeom(settings.width, settings.height);

    setBackground(screen);
    setDirty(screen, background);

    return true;
}
EventsReturn process_events(GameState* gameState, GameClock& gameClock)
{
    SDL_Event event;
    while (SDL_PollEvent(&event))
    {
	if (event.type == SDL_QUIT)
	    return ER_QUIT;
	else if (event.type == SDL_VIDEORESIZE)
	{
	    settings.width = event.resize.w;
	    settings.height = event.resize.h;
	    setVideoMode();
	    return ER_MISC;
	}
	else if (event.type == SDL_KEYDOWN)
	{
	    //Key key(event.key.keysym);
	    switch (event.key.keysym.sym) {
		case SDLK_q: return ER_QUIT;
		case SDLK_p: gameClock.paused = !gameClock.paused; break;
		case SDLK_r: return ER_RESTART;
		case SDLK_MINUS: gameClock.rate = 4*gameClock.rate/5; break;
		case SDLK_EQUALS: case SDLK_PLUS: gameClock.rate = std::max(5*gameClock.rate/4,
					 gameClock.rate+1); break;
		case SDLK_b: if (settings.bgType == BG_LAST)
				 settings.bgType = BG_FIRST;
			     else
				 settings.bgType = BGType(settings.bgType + 1);
			     return ER_NEWBACKGROUND;
		case SDLK_1: settings.N = max(1, settings.N-1); return ER_RESTART;
		case SDLK_2: settings.N++; return ER_RESTART;
		case SDLK_s: settings.stochastic = !settings.stochastic; return ER_RESTART;
		case SDLK_g: settings.graph = !settings.graph; return ER_RESTART;
		case SDLK_d: settings.dampen = !settings.dampen;
			     gameState->toggleDamping();
			     break;
		case SDLK_a: settings.arrows = !settings.arrows;
			     gameState->arrows = settings.arrows;
			     break;
		case SDLK_l: settings.linear = !settings.linear;
			     gameState->linear = settings.linear;
			     break;
		default:;
	    }
	}
    }
    return ER_NONE;
}


void initialize_system()
{
    /* Initialize SDL */
    if ( SDL_Init(SDL_INIT_EVERYTHING) < 0 ) {
	fprintf(stderr,
		"Couldn't initialize SDL: %s\n", SDL_GetError());
	exit(1);
    }
    atexit(SDL_Quit);			/* Clean up on exit */

    SDL_EnableKeyRepeat(SDL_DEFAULT_REPEAT_DELAY, SDL_DEFAULT_REPEAT_INTERVAL);

    // set random seed
    srand(time(NULL));
}

void initialize_video()
{
    /* Initialize the display */
    if (settings.width != 0 && settings.height != 0)
    {
	// try the mode we've been given
	screen = SDL_SetVideoMode(settings.width, settings.height,
		settings.bpp, settings.videoFlags);
	if ( screen == NULL ) {
	    fprintf(stderr, "Couldn't set %dx%dx%d video mode: %s\n",
		    settings.width, settings.height, settings.bpp,
		    SDL_GetError());
	}
    }

    if (screen == NULL)
    {
	// try a mode SDL tells us shold work:
	SDL_Rect** modes = SDL_ListModes(NULL, settings.videoFlags);
	if (modes == NULL)
	{
	    fprintf(stderr,
		    "SDL reports no modes available\n");
	    exit(1);
	}
	if (modes == (SDL_Rect**)(-1))
	{
	    // "All modes available"
	    settings.width = 1024;
	    settings.height = 768;
	}
	else
	{
	    // use the first (i.e. biggest) mode which is no bigger than
	    // 1024x768
	    SDL_Rect* bestMode;
	    do
	    {
		bestMode = *modes;
		if (bestMode->w <= 1024 && bestMode->h <= 768)
		    break;
	    } while (++modes);
	    settings.width = bestMode->w;
	    settings.height = bestMode->h;
	}

	screen = SDL_SetVideoMode(settings.width, settings.height,
		settings.bpp, settings.videoFlags);
	if ( screen == NULL ) {
	    fprintf(stderr, "Couldn't set %dx%dx%d video mode: %s\n",
		    settings.width, settings.height, settings.bpp,
		    SDL_GetError());

	    // one last try... let SDL use any bpp:
	    screen = SDL_SetVideoMode(settings.width, settings.height,
		    settings.bpp, settings.videoFlags | SDL_ANYFORMAT);
	    if ( screen == NULL )
	    {
		fprintf(stderr, "Couldn't set %dx%d video mode: %s\n",
			settings.width, settings.height, SDL_GetError());
		// give up
		exit(1);
	    }
	}
    }

    /* Show some info */
    printf("Set %dx%dx%d mode\n",
	    screen->w, screen->h, screen->format->BitsPerPixel);
    printf("Video surface located in %s memory.\n",
	    (screen->flags&SDL_HWSURFACE) ? "video" : "system");

    SDL_ShowCursor(SDL_DISABLE);

    if ( !initFont() )
    {
	fprintf(stderr, "Failed to load font data file\n");
	exit(1);
    }

    screenGeom = ScreenGeom(settings.width, settings.height);

    setBackground(screen);
    setDirty(screen, background);
}

void run_game()
{
    GameState* gameState = new GameState(settings.N, settings.stochastic,
	    settings.graph, settings.dampen, settings.arrows, settings.linear);
    GameClock gameClock(rateOfSpeed(settings.speed));

    // main loop
    Uint32 lastStateUpdate = 0, beforeDelay = 0;
    Uint32 ticksBefore, ticksAfter;
    Uint32 loopTicks = SDL_GetTicks();
    int timeTillNextFrame = 0;
    int delayTime, actualDelayed, updateTime;
    float avFrameTime = 1000/settings.fps;
    int fpsRegulatorTenthTicks = 0;
    const int avFrames = 10; // number of frames to average over
    EventsReturn eventsReturn = ER_NONE;
    bool wantScreenshot = false;
    bool wantVideo = false;
    int videoFrame = 1;
    bool quit = false;
    bool forceFrame = false;

    const int MIN_INPUT_STEP = 30;
    const int MIN_GAME_STEP = 30;

    lastStateUpdate = SDL_GetTicks();
    while ( !quit ) {
	forceFrame = false;
	while (timeTillNextFrame > 0)
	{
	    delayTime = std::min(timeTillNextFrame, MIN_INPUT_STEP);
	    beforeDelay = SDL_GetTicks();
	    SDL_Delay(delayTime);
	    actualDelayed = SDL_GetTicks() - beforeDelay;
	    timeTillNextFrame -= actualDelayed;

	    eventsReturn = process_events(gameState, gameClock);

	    switch (eventsReturn)
	    {
		case ER_RESTART:
		    {
			GameState* newGameState =
			    new GameState(settings.N, settings.stochastic,
				    settings.graph, settings.dampen,
				    settings.arrows, settings.linear);
			delete gameState;
			gameState = newGameState;
			gameClock = GameClock(rateOfSpeed(settings.speed));
		    }
		    lastStateUpdate = SDL_GetTicks();
		    break;
		case ER_QUIT:
		    quit = true;
		    break;
		case ER_SCREENSHOT:
		    wantScreenshot = true;
		    break;
		case ER_NEWBACKGROUND:
		    if (!background)
		    {
			setBackground(screen);
			setDirty(screen, background);
		    }
		    else
			drawBackground(screen);
		    forceFrame = true;
		    break;
		case ER_NOTIMETAKEN:
		    // pretend the time spent in process_events() didn't
		    // actually happen:
		    lastStateUpdate = SDL_GetTicks();
		    forceFrame = true;
		    break;
		case ER_MISC:
		    forceFrame = true;
		    break;
		default: ;
	    }

	    updateTime = std::max(1,
		    gameClock.scale(SDL_GetTicks() - lastStateUpdate));
	    lastStateUpdate = SDL_GetTicks();
	    if ( !gameClock.paused )
	    {
		while (updateTime > 0)
		{
		    const int stepTime =
			std::min( MIN_GAME_STEP, updateTime );
		    gameState->update(screen, stepTime);
		    gameClock.updatePreScaled(stepTime);
		    updateTime -= stepTime;
		}
	    }
	}

	ticksBefore = SDL_GetTicks();
	if (!gameClock.paused || forceFrame)
	{
	    gameState->draw(screen);
	    //drawInfo(screen, gameState, gameClock, 1000.0/avFrameTime);
	    SDL_Flip(screen);

	    if (wantScreenshot)
	    {
		if (SDL_SaveBMP(screen, "screenshot.bmp") != 0)
		    fprintf(stderr, "Screenshot failed.\n");
		wantScreenshot = false;
	    }
	    if (wantVideo)
	    {
		char fname[16];
		snprintf(fname, 16, "frame%.5d.bmp", videoFrame);
		SDL_SaveBMP(screen, fname);
		videoFrame++;
	    }

	    // blank over what we've drawn:
	    blankDirty();
	}
	ticksAfter = SDL_GetTicks();

	const int renderingTicks = ticksAfter - ticksBefore;

	// Add in a manual tweak to the delay, to ensure we approximate the
	// requested fps (where possible):
	if (10000/avFrameTime < settings.fps * 10 &&
		1000/settings.fps - renderingTicks + fpsRegulatorTenthTicks/10
		> 1)
	    fpsRegulatorTenthTicks -= 1;
	else if (10000/avFrameTime > settings.fps * 10)
	    fpsRegulatorTenthTicks += 1;

	timeTillNextFrame =
	    std::max(1, (1000/settings.fps) - renderingTicks +
		    fpsRegulatorTenthTicks/10
		    //+ (rani(10) < fpsRegulatorTenthTicks%10)
		    );

	const int loopTime = SDL_GetTicks() - loopTicks;
	avFrameTime += (loopTime-avFrameTime)/avFrames;
	loopTicks = SDL_GetTicks();
    }

    delete gameState;

    SDL_Quit();
}

int main(int argc, char** argv)
{
    load_settings(argc, argv);
    initialize_system();
    initialize_video();
    run_game();

    return 0;
}
