/*
 * Kuklomenos
 * Copyright (C) 2008-2009 Martin Bays <mbays@sdf.lonestar.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see http://www.gnu.org/licenses/.
 */

#include <vector>
#include <set>
#include <algorithm>
#include <cstdio>
#include <SDL/SDL.h>
#include <SDL_gfxPrimitivesDirty.h>

#include "state.h"
#include "geom.h"
#include "gfx.h"
#include "settings.h"
#include "data.h"

const int MAXN=1024;

// HSVtoRGB nabbed from qonk, Copyright 2005 by Anthony Liekens
// anthony@liekens.net, GPLv2
Uint32 HSVtoRGB( float h, float s, float v )
{
	int i;
	float f, p, q, t, r, g, b;

	if( s == 0 ) {

		// achromatic (grey)
		r = g = b = v;

	} else {

		h /= 60;			// sector 0 to 5
		i = (int)h;
		f = h - i;			// factorial part of h
		p = v * ( 1 - s );
		q = v * ( 1 - s * f );
		t = v * ( 1 - s * ( 1 - f ) );
	
		switch( i ) {
			case 0:
				r = v;
				g = t;
				b = p;
				break;
			case 1:
				r = q;
				g = v;
				b = p;
				break;
			case 2:
				r = p;
				g = v;
				b = t;
				break;
			case 3:
				r = p;
				g = q;
				b = v;
				break;
			case 4:
				r = t;
				g = p;
				b = v;
				break;
			default:		// case 5:
				r = v;
				g = p;
				b = q;
			break;
		}

	}

	return 256 * (256 * 256 * (int)( 255 * r ) + 256 * (int)( 255 * g ) + (int)( 255 * b ));
}

Uint32 GameState::colourOfState(int i)
{
    return HSVtoRGB(int(30 + 360.0*i/N)%360, 0.9, 0.9);
}

bool GameState::inputMatrixEntry(SDL_Surface* surface, int i, int j)
{
    // quick-and-dirty text entry:
    int pos=0;
    const int maxlen=10;
    char text[maxlen+1];
    text[pos] = '\0';
    SDL_Event event;
    float *p = j>=N ? &state[i] : &matrix[i][j];
    float orig = *p;
    int done = false;
    SDLKey sym;
    char c;
    while (!done)
    {
	SDL_Delay(50);
	int redraw = true;
	while (SDL_PollEvent(&event))
	    switch (event.type)
	    {
		case SDL_QUIT:
		    return false;
		case SDL_KEYDOWN:
		    sym = event.key.keysym.sym;
		    c=0;
		    if (0 <= sym - SDLK_0 && sym - SDLK_0 < 10 &&
			    !(event.key.keysym.mod & KMOD_SHIFT))
			c = (sym - SDLK_0) + '0';
		    else if (sym == SDLK_PERIOD)
			c = '.';
		    else if (sym == SDLK_MINUS)
			c = '-';
		    if (c)
		    {
			if (pos < maxlen)
			{
			    text[pos++] = c;
			    text[pos] = '\0';
			    redraw = true;
			}
		    }

		    else if ( (sym == SDLK_DELETE || sym == SDLK_BACKSPACE) &&
			    pos > 0 )
		    {
			text[--pos] = '\0';
			redraw = true;
		    }

		    else if (sym == SDLK_ESCAPE) {
			*p = orig;
			return false;
		    }

		    else if (sym == SDLK_RETURN)
			done = true;

		    break;

		case SDL_MOUSEBUTTONDOWN:
		    done = true;
		    break;
	    }
	bool ret = sscanf(text, "%f", p);
	if (!ret)
	    *p=0.0;
	if (redraw)
	{
	    hi=i;
	    hj=j;
	    draw(surface);
	    SDL_Flip(surface);
	    blankDirty();
	    redraw = false;
	}
    }
    return true;
}
void GameState::update(SDL_Surface* surface, int time)
{
    if (time <= 0)
	return;

    int x, y;
    const int buttons = SDL_GetMouseState(&x,&y);
    const bool rbutton = SDL_BUTTON_RMASK & buttons;
    const bool lbutton = SDL_BUTTON_LMASK & buttons;
    const bool shift = SDL_GetModState() & (KMOD_LSHIFT | KMOD_RSHIFT);
    if (lbutton || rbutton) {
	CartCoord cursor(x-screenGeom.centre.x,-(y-screenGeom.centre.y));
	/*
	int mj = (cursor.x + ((N-1)/2.0 + 0.5)*5*10) / (5*10);
	int mi = (-cursor.y + ((N-1)/2.0 + 0.5)*30) / 30;
	*/
	int mj = (x - 20) / (6*10);
	int mi = (y - 20) / 30;
	if (0 <= mi && mi < N && 0 <= mj && (mj < N || mj == N+2)) {
	    if (lbutton && rbutton) {
		inputMatrixEntry(surface,mi,mj);
		hi=hj=-1;
	    }
	    else {
		float amount = (shift ? 1 : 5) *
		    (rbutton ? -1 : 1) * 
		    time/10/1000.0;
		if (mj<N)
		    matrix[mi][mj] += amount;
		else
		    state[mi] += amount;
		if (stochastic)
		    stochastify();
	    }
	}
	else {
	    RelPolarCoord c(cursor - ARENA_CENTRE);
	    if (5*ARENA_RAD/3.0 > c.dist && c.dist > ARENA_RAD/3.0) {
		int i = (int)floor( (N/4.0)*(c.angle+1) + 0.5) % N;
		if (graph && rbutton && !lbutton) {
		    if (adjfrom == -1)
			adjfrom = i;
		    else if (adjfrom != i) {
			adjMatrix[i][adjfrom] = 1-adjMatrix[i][adjfrom];
			calcGraphMatrix();
			adjfrom = -1;
		    }
		}
		else {
		    float amount = (shift ? 1 : 5) *
			(rbutton ? -1 : 1) * 
			time/10/1000.0;
		    if (stochastic) {
			amount = max(amount, -state[i]);
			amount = min(amount, 1-state[i]);
			for (int i2=0; i2<N; i2++)
			    if (i2 != i)
				state[i2] -= amount/(N-1);
		    }
		    state[i] += amount;
		    if (stochastic)
			stochastify();
		}
	    }
	}
    }
    else {
	adjfrom = -1;
	if (SDL_GetKeyState(NULL)[SDLK_SPACE]) {
	    t -= time;
	    while (t < 0) {
		applyMatrix();
		t += period;
	    }
	}
	if (SDL_GetKeyState(NULL)[SDLK_PERIOD])
	    angle += time/1000.0;
	if (SDL_GetKeyState(NULL)[SDLK_COMMA])
	    angle -= time/1000.0;
    }

    if (time >= 200)
	zoomRad = targetZoomRad;
    else
	zoomRad += (targetZoomRad-zoomRad)*time/200;
}

void GameState::calcGraphMatrix(){
    float dampening = dampen ? 0.1 : 0.0;
    for (int j=0; j<N; j++) {
	int count=0;
	for (int i=0; i<N; i++)
	    count += adjMatrix[i][j];
	for (int i=0; i<N; i++) {
	    if (i==j)
		matrix[i][j]=0.0;
	    else {
		if (adjMatrix[i][j])
		    matrix[i][j] = dampening/(N-1) + (1.0-dampening)/count;
		else
		    matrix[i][j] = (count==0 ? 1.0 : dampening)/(N-1);
	    }
	}
    }
}
void GameState::stochastify() {
    for (int j=0; j<N; j++) {
	float total=0;
	for (int i=0; i<N; i++) {
	    matrix[i][j] = min(1.0f, max(0.0f, matrix[i][j]));
	    total += matrix[i][j];
	}
	for (int i=0; i<N; i++)
	    matrix[i][j] /= total;
    }
    float total=0;
    for (int i=0; i<N; i++) {
	state[i] = min(1.0f, max(0.0f, state[i]));
	total += state[i];
    }
    for (int i=0; i<N; i++)
	state[i] /= total;
}
void GameState::applyMatrix()
{
    float newstate[MAXN];
    for (int i=0; i<N; i++) {
	newstate[i]=0;
	for (int j=0; j<N; j++)
	    newstate[i] += matrix[i][j] * state[j];
    }
    for (int i=0; i<N; i++)
	state[i]=newstate[i];
}

void GameState::draw(SDL_Surface* surface)
{
    const View view(ARENA_CENTRE,
		(double)screenGeom.rad/(double)zoomRad,
		angle) ;

    //Circle(ARENA_CENTRE, ARENA_RAD,
	    //0x808080ff).draw(surface, view, NULL, true);
    
    gfxPrimitivesSetFont(fontBig, 10, 20);
    float dispRad = ARENA_RAD;
    if (linear)
	    Line(ARENA_CENTRE + RelCartCoord(0, zoomRad),
		    ARENA_CENTRE - RelCartCoord(0, zoomRad),
		    0x00ff0080).draw(surface,view);
    for (int i=0; i<N; i++)
	if (linear) {
	    const int dotRad = 10 * zoomRad/ARENA_RAD;
	    const RelCartCoord d(ARENA_RAD*state[i],
		    (2*(N/2-i)-((N+1)%2))*zoomRad/(N+1) );
	    Line(ARENA_CENTRE + RelCartCoord(-1.2*zoomRad, d.dy),
		    ARENA_CENTRE + RelCartCoord(1.2*zoomRad, d.dy),
		    0x00ff0080).draw(surface,view);
	    float markDist = 1;
	    while (zoomRad/ARENA_RAD + 1 > markDist) {
		for (int sign=1; sign != -3; sign -= 2)
		    Line(ARENA_CENTRE + RelCartCoord(sign*markDist*ARENA_RAD, d.dy-5*markDist),
			    ARENA_CENTRE + RelCartCoord(sign*markDist*ARENA_RAD, d.dy+5*markDist),
			    0x00ff0080).draw(surface,view);
		markDist *= 10;
	    }
	    Circle(ARENA_CENTRE + d, dotRad,
		    colourOfState(i)|0xb0, true).draw(surface,view);
	    Circle(ARENA_CENTRE + d, dotRad,
		    colourOfState(i)|0xff).draw(surface,view);
	    dispRad = max(dispRad, RelPolarCoord(d).dist);
	}
	else {
	    const float chainRad = ARENA_RAD/2;
	    const CartCoord centre = ARENA_CENTRE + RelPolarCoord(i*4.0/N-1, chainRad);
	    const float rad = PI*ARENA_RAD*sqrt(state[i])/(2*N);
	    const CartCoord above = centre + RelCartCoord(0,max(0.0f,rad));
	    const ScreenCoord aboveScr = view.coord(above);
	    Circle(centre, rad,
		    colourOfState(i)|0xb0, true).draw(surface,view);
	    Circle(centre, rad,
		    colourOfState(i)|0xff).draw(surface,view);
	    char s[15];
	    snprintf(s, 15, "%d: %.2f", i+1, state[i]);
	    stringColor(surface, aboveScr.x-10*strlen(s)/2, aboveScr.y-20,
		    s, colourOfState(i)|0xff);
	    dispRad = max(dispRad, rad+chainRad);
	}

    targetZoomRad = dispRad;

    if (arrows && !linear)
	for (int i=0; i<N; i++) {
	    const CartCoord icentre = ARENA_CENTRE + RelPolarCoord(i*4.0/N-1, ARENA_RAD/2);
	    //const float irad = PI*ARENA_RAD*sqrt(state[i])/(2*N);
	    for (int j=0; j<N; j++)
		if (i != j && abs(matrix[i][j]) > 0.05) {
		    // draw arrows:
		    const CartCoord jcentre = ARENA_CENTRE + RelPolarCoord(j*4.0/N-1, ARENA_RAD/2);
		    //const float jrad = PI*ARENA_RAD*sqrt(state[j])/(2*N);
		    const RelPolarCoord d = icentre - jcentre;
		    const RelPolarCoord normal =  d.rotated(1) * (1/d.dist) ;
		    const float thickness = 10*abs(matrix[i][j]);
		    const Uint32 col = matrix[i][j] > 0 ? 0xffffffa0 : 0xff8080a0;
		    CartCoord points[5];
		    points[0] = jcentre + d*(1.0/4) + normal*(20 + thickness/2);
		    points[1] = icentre + d*(-3.0/16) + normal*(20 + thickness/2);
		    points[2] = icentre + d*(-1.0/8) + normal*(20);
		    points[3] = icentre + d*(-3.0/16) + normal*(20 - thickness/2);
		    points[4] = jcentre + d*(1.0/4) + normal*(20 -thickness/2);
		    Polygon(points, 5, col, true).draw(surface, view);
		    Line( icentre + d*(-1.0/8) + normal*20,
			    icentre + d*(-1.0/4) + normal*(20 + thickness),
			    col|0xff).draw(surface, view);
		    Line( icentre + d*(-1.0/8) + normal*20,
			    icentre + d*(-1.0/4) + normal*(20 - thickness),
			    col|0xff).draw(surface, view);
		}
	}

    drawMatrix(surface);

    const View screenView(0, 1, 0);
    int x, y;
    SDL_GetMouseState(&x,&y);
    CartCoord cursor(x-screenGeom.centre.x,-(y-screenGeom.centre.y));
    Circle(cursor, 5, 0xff0000a0, true).draw(surface,screenView);
    Circle(cursor, 5, 0xff0000ff).draw(surface,screenView);
}
void GameState::drawMatrix(SDL_Surface* surface) {
    const View screenView(0, 1, 0);
    for (int i=0; i<N; i++)
	for (int j=0; j<N; j++) {
	    char s[6];
	    snprintf(s, 6, "%.2f", matrix[i][j]);
	    /*
	    CartCoord pos((j-(N-1)/2.0 - 0.5)*5*10, -(i-(N-1)/2.0)*30);
	    ScreenCoord scrpos = screenView.coord(pos);
	    stringColor(surface, scrpos.x, scrpos.y, s, 0xffffffa0);
	    */
	    stringColor(surface, 20+j*6*10, 20+i*30, s, 0xffffffa0 | ((i==hi && j==hj) ? 0xff : 0));
	}

    for (int i=0; i<N; i++) {
	    char s[12];
	    snprintf(s, 12, "%.2f", state[i]);
	    stringColor(surface, 20+(N+2)*6*10, 20+i*30, s, 0xffffffa0 | ((i==hi && hj>=N) ? 0xff : 0));
    }
}
